// Middleware independent class to do processing

#ifndef IMAGEPROCESSING_HPP
#define IMAGEPROCESSING_HPP

//C++ Includes
#include <iostream>
#include <vector>

//OS Includes
#include <unistd.h>
#include <mutex>

namespace UILLINOIS
{
namespace Perception
{

typedef struct
{
  float x;
  float y;
  float z;
  float intensity;
  float ring;

} point_t;

class ImageProcessing
{
public:
  ImageProcessing();

  ~ImageProcessing();

  // Called to pass in parameters and initialize
  bool init(double config1_in);

  // Called at desired rate
  void update(double now);

  // Called when image data is received
  void receiveImage(double time,
                    unsigned int height_in,
                    unsigned int width_in,
                    unsigned int step_in,
                    const std::vector<unsigned char> &data_in);

  // Called when a point cloud is received
  void receiveLidar(double time,
                    unsigned int x_offset_in,
                    unsigned int y_offset_in,
                    unsigned int z_offset_in,
                    unsigned int intensity_offset,
                    unsigned int ring_offset_in,
                    bool big_endian_in,
                    unsigned int point_size_in,
                    const std::vector<unsigned char> &data_in);

  // Called when an imu message is received
  void receiveIMU(double time,
                  double roll,
                  double pitch);

  // Called when a vehicle velocity message is received
  void receiveVelocity(double time,
                       double vehicle_velocity);

  // Provides some output like "ok to change lanes"
  // Returns true if output is valid
  bool getOkToChangeLane(bool &ok_to_change_lane);

private:
  void do_processing();

  // Constants
  const double config1_min_ = 1.0;
  const double config1_max_ = 2.0;

  // Configurations
  double config1_;

  // Mutexes
  std::mutex msg_mutex_;

  // Inputs
  unsigned int image_height_;
  unsigned int image_width_;
  unsigned int image_step_;
  std::vector<unsigned char> image_data_;
  double last_image_msg_time_;

  std::vector<point_t> lidar_points_;
  double last_lidar_msg_time_;

  double pitch_;
  double roll_;
  double last_imu_msg_time_;

  double vehicle_velocity_;
  double last_velocity_msg_time_;

  // Outputs
  bool ok_to_change_lane_;
  bool ok_to_change_lane_valid_;

  // Intermediata variables
  double last_image_msg_time_processed_;
};

}
}
#endif
