
#include <image_processing.hpp>

#include <ros/ros.h>

#include <sensor_msgs/image_encodings.h>
#include <sensor_msgs/Image.h>
#include <sensor_msgs/PointCloud2.h>
#include <novatel_msgs/INSPVAX.h>
#include <std_msgs/Float32.h>
#include <std_msgs/Bool.h>

using namespace std;
using namespace UILLINOIS::Perception;

// Create instance of class here so that call back functions can use it
ImageProcessing module;

// Call back functions
void imageCallback(const sensor_msgs::Image::ConstPtr& msg)
{
  module.receiveImage(msg->header.stamp.toSec(),
                      msg->height,
                      msg->width,
                      msg->step,
                      msg->data);
}

void lidarCallback(const sensor_msgs::PointCloud2::ConstPtr& msg)
{
  // Probably should be reading offsets out of msg->fields
  module.receiveLidar(msg->header.stamp.toSec(), 0, 4, 8, 16, 20, msg->is_bigendian > 0, msg->point_step, msg->data);
}

void imuCallback(const novatel_msgs::INSPVAX::ConstPtr& msg)
{
  module.receiveIMU(ros::Time::now().toSec(), msg->roll, msg->pitch);
}

void velCallback(const std_msgs::Float32::ConstPtr& msg)
{
  module.receiveVelocity(ros::Time::now().toSec(), msg->data);
}

// Main routine
int main(int argc, char **argv)
{
  bool exit = false;

  double publish_interval = 0.02;

  if (!module.init(1.4))
  {
    exit = true;
  }

  if (exit)
    return 0;

  // ROS initialization
  ros::init(argc, argv, "image_processing");
  ros::NodeHandle n;
  ros::Rate loop_rate(1.0 / publish_interval);

  // Advertise messages to send
  ros::Publisher ok_to_change_lane_pub = n.advertise<std_msgs::Bool>("ok_to_change_lane", 1);
  std_msgs::Bool ok_to_change_lane_msg;

  // Subscribe to messages to read
  ros::Subscriber image_sub = n.subscribe("/camera/image_raw", 1, imageCallback);
  ros::Subscriber lidar_sub = n.subscribe("/velodyne_points", 1, lidarCallback);
  ros::Subscriber imu_sub = n.subscribe("/novatel_data/inspvax", 1, imuCallback);
  ros::Subscriber vel_sub = n.subscribe("/vehicle_speed", 1, velCallback);

  // Start asynchronous spinner
//  ros::AsyncSpinner spinner(4);
//  spinner.start();

  // Wait for time to be valid
  while (ros::Time::now().nsec == 0);

  // Loop as long as module should run
  while (ros::ok())
  {
    // Get current time
    ros::Time now = ros::Time::now();
    double now_sec = now.toSec();

    module.update(now_sec);

    bool ok;
    if (module.getOkToChangeLane(ok))
    {
      ok_to_change_lane_msg.data = (unsigned char) (ok ? 1 : 0);
      ok_to_change_lane_pub.publish(ok_to_change_lane_msg);
    }

    // Wait for next loop
    loop_rate.sleep();

    // Receive messages here if not using AsyncSpinner
    ros::spinOnce();
  }

  // Stop receiving all messages
//  spinner.stop();

  return 0;
}

