// Middleware independent class to do processing

#include <image_processing.hpp>

using namespace std;
using namespace UILLINOIS::Perception;

//Default constructor.
ImageProcessing::ImageProcessing() :
  msg_mutex_(),
  image_height_(),
  image_width_(),
  image_step_(),
  image_data_(),
  last_image_msg_time_(0.0),
  lidar_points_(),
  last_lidar_msg_time_(0.0),
  pitch_(),
  roll_(),
  last_imu_msg_time_(0.0),
  vehicle_velocity_(),
  last_velocity_msg_time_(0.0),
  ok_to_change_lane_(false),
  ok_to_change_lane_valid_(false),
  last_image_msg_time_processed_(0.0)
{
}

//Default destructor.
ImageProcessing::~ImageProcessing()
{
}

bool ImageProcessing::init(double config1_in)
{
  bool config_ok = true;

  if ((config1_in >= config1_min_) && (config1_in <= config1_max_))
  {
    config1_ = config1_in;
  }
  else
  {
    config_ok = false;
  }

  return config_ok;
}

void ImageProcessing::update(double now)
{
  (void) now;

  msg_mutex_.lock();
  double last_image_msg_time = last_image_msg_time_;
  msg_mutex_.unlock();

  if (last_image_msg_time_processed_ != last_image_msg_time)
  {
    do_processing();
    last_image_msg_time_processed_ = last_image_msg_time;
  }
}

void ImageProcessing::receiveImage(double time,
                                   unsigned int height,
                                   unsigned int width,
                                   unsigned int step,
                                   const vector<unsigned char> &data)
{
  msg_mutex_.lock();
  last_image_msg_time_ = time;

  image_height_ = height;
  image_width_ = width;
  image_step_ = step;
  image_data_ = data;

  msg_mutex_.unlock();
}

void ImageProcessing::receiveLidar(double time,
                                   unsigned int x_offset,
                                   unsigned int y_offset,
                                   unsigned int z_offset,
                                   unsigned int intensity_offset,
                                   unsigned int ring_offset,
                                   bool big_endian,
                                   unsigned int point_size,
                                   const std::vector<unsigned char> &data)
{
  msg_mutex_.lock();
  last_lidar_msg_time_ = time;

  lidar_points_.clear();
  point_t point;
  const unsigned char *data_ptr = &data[0];
  for (unsigned int idx = 0; idx < data.size(); idx += point_size)
  {
    float x, y, z;
    float intensity;
    unsigned short ring;

    if (big_endian)
    {
      unsigned int temp32;
      temp32 = ((unsigned int) data_ptr[x_offset] << 24) +
               ((unsigned int) data_ptr[x_offset + 1] << 16) +
               ((unsigned int) data_ptr[x_offset + 2] << 8) +
               (unsigned int) data_ptr[x_offset + 3];
      x = *(reinterpret_cast<float *>(&temp32));

      temp32 = ((unsigned int) data_ptr[y_offset] << 24) +
               ((unsigned int) data_ptr[y_offset + 1] << 16) +
               ((unsigned int) data_ptr[y_offset + 2] << 8) +
               (unsigned int) data_ptr[y_offset + 3];
      y = *(reinterpret_cast<float *>(&temp32));

      temp32 = ((unsigned int) data_ptr[z_offset] << 24) +
               ((unsigned int) data_ptr[z_offset + 1] << 16) +
               ((unsigned int) data_ptr[z_offset + 2] << 8) +
               (unsigned int) data_ptr[z_offset + 3];
      z = *(reinterpret_cast<float *>(&temp32));

      temp32 = ((unsigned int) data_ptr[intensity_offset] << 24) +
               ((unsigned int) data_ptr[intensity_offset + 1] << 16) +
               ((unsigned int) data_ptr[intensity_offset + 2] << 8) +
               (unsigned int) data_ptr[intensity_offset + 3];
      intensity = *(reinterpret_cast<float *>(&temp32));

      ring = ((unsigned short) data_ptr[ring_offset] << 8) +
             (unsigned short) data_ptr[ring_offset + 1];
    }
    else
    {
      unsigned int temp32;
      temp32 = ((unsigned int) data_ptr[x_offset + 3] << 24) +
               ((unsigned int) data_ptr[x_offset + 2] << 16) +
               ((unsigned int) data_ptr[x_offset + 1] << 8) +
               (unsigned int) data_ptr[x_offset];
      x = *(reinterpret_cast<float *>(&temp32));

      temp32 = ((unsigned int) data_ptr[y_offset + 3] << 24) +
               ((unsigned int) data_ptr[y_offset + 2] << 16) +
               ((unsigned int) data_ptr[y_offset + 1] << 8) +
               (unsigned int) data_ptr[y_offset];
      y = *(reinterpret_cast<float *>(&temp32));

      temp32 = ((unsigned int) data_ptr[z_offset + 3] << 24) +
               ((unsigned int) data_ptr[z_offset + 2] << 16) +
               ((unsigned int) data_ptr[z_offset + 1] << 8) +
               (unsigned int) data_ptr[z_offset];
      z = *(reinterpret_cast<float *>(&temp32));

      temp32 = ((unsigned int) data_ptr[intensity_offset + 3] << 24) +
               ((unsigned int) data_ptr[intensity_offset + 2] << 16) +
               ((unsigned int) data_ptr[intensity_offset + 1] << 8) +
               (unsigned int) data_ptr[intensity_offset];
      intensity = *(reinterpret_cast<float *>(&temp32));

      ring = ((unsigned short) data_ptr[ring_offset + 1] << 8) +
             (unsigned short) data_ptr[ring_offset];
    }
    data_ptr += point_size;

    point.x = x;                   // Meters, positive is forward
    point.y = y;                   // Meters, positive is left
    point.z = z;                   // Meters, positive is up
    point.intensity = intensity;   // not sure?
    point.ring = ring;             // Which laser this point is from (0 to 31)
    lidar_points_.push_back(point);
  }

  msg_mutex_.unlock();
}

void ImageProcessing::receiveIMU(double time,
                                 double roll,
                                 double pitch)
{
  msg_mutex_.lock();
  last_imu_msg_time_ = time;
  roll_ = roll;     // Degrees, rolling to the right is positive
  pitch_ = pitch;   // Degress, pitching up is positive
  msg_mutex_.unlock();
}

void ImageProcessing::receiveVelocity(double time,
                                      double vehicle_velocity)
{
  msg_mutex_.lock();
  last_velocity_msg_time_ = time;
  vehicle_velocity_ = vehicle_velocity;  // Meters/sec
  msg_mutex_.unlock();
}

bool ImageProcessing::getOkToChangeLane(bool &ok_to_change_lane)
{
  if (!ok_to_change_lane_valid_)
    return false;

  ok_to_change_lane = ok_to_change_lane_;

  return true;
}

void ImageProcessing::do_processing()
{
  // Make local copies
  msg_mutex_.lock();
  unsigned int image_height = image_height_;
  unsigned int image_width = image_width_;
  unsigned int image_step = image_step_;
  std::vector<unsigned char> image_data = image_data_;

  std::vector<point_t> lidar_points = lidar_points_;

  double pitch = pitch_;
  double roll = roll_;

  double vehicle_velocity = vehicle_velocity_;
  msg_mutex_.unlock();

  // MAGIC HAPPENS

  ok_to_change_lane_ = true;
  ok_to_change_lane_valid_ = true;
}
